# ADOBE CONFIDENTIAL
#
# Copyright 2010-2021 Adobe
# All Rights Reserved.
#
# NOTICE:  Adobe permits you to use, modify, and distribute this file in
# accordance with the terms of the Adobe license agreement accompanying it.
# If you have received this file from a source other than Adobe,
# then your use, modification, or distribution of it requires the prior
# written permission of Adobe.
#

import sys
import substance
import xml.etree.ElementTree as ET

#==============================================================================
class ReturnValue:
    def __init__(self):
        self.mDict = {}

    def getValue(self, aName):
        return self.mDict[aName]

    def setValue(self, aName, aValue):
        self.mDict[aName] = aValue

    def getKeys(self):
        return self.mDict.keys()

    def getValueDict(self):
        return self.mDict

#==============================================================================
class CommandLineArgs:
    def __init__(self):
        self.mArgs = {}

    def hasArgument(self, aArgName):
        if aArgName in self.mArgs:
            return True
        return False

    def getArgument(self, aArgName):
        if aArgName in self.mArgs:
            return self.mArgs[aArgName]
        return None

    def parseArgs(self, aArgs):
        def isArgument(aStr):
            if aStr[0] == '-' and aStr[len(aStr)-1] == ':':
                return True
            return False
        #======================================================================
        def getArgumentName(aStr):
            return aStr[1:len(aStr)-1]
        #======================================================================
        i = 0

        self.mArgs = {}
        argsLen = len(aArgs)
        while i < argsLen:
            arg = aArgs[i]
            i += 1

            if arg is None:
                continue

            argLen = len(arg)
            if argLen == 0:
                continue

            if isArgument(arg):
                argName = getArgumentName(arg)
                # Add argument
                self.mArgs[argName] = True

                if i < argsLen:
                    nextArg = aArgs[i]
                    if not isArgument(nextArg):
                        self.mArgs[argName] = nextArg
                        i += 1

        # print('Args:' + str(self.mArgs))

#==============================================================================
class ArgsFile:
    def __init__(self, aFileAbsPath):
        self.mFileAbsPath = aFileAbsPath
        self.mXMLRoot = None
        self.mSBSContext = substance.Context()
        self.mFctArgs = {}

    def getSBSContext(self):
        return self.mSBSContext

    def getFunctionArgs(self):
        return self.mFctArgs

    def parse(self):
        # Get fctArgs from file:
        try:
            tree = ET.parse(self.mFileAbsPath)
            self.mXMLRoot = tree.getroot()
            if self.mXMLRoot is None:
                return False
        except :
            print('Fail to parse file')
            return False

        # Build SBS Context
        self.mSBSContext = substance.Context()
        xmlElmtContext = self.mXMLRoot.find('context')
        if xmlElmtContext is not None:
            # URl Aliases
            xmlElmtUrlAliases = xmlElmtContext.find('url_aliases')
            for xmlElmtChild in xmlElmtUrlAliases:
                if not xmlElmtChild.tag == 'url_alias':
                    continue
                aliasName = xmlElmtChild.get('name')
                if aliasName is None:
                    continue
                aliasPath = xmlElmtChild.get('path')
                if aliasPath is None:
                    continue
                self.mSBSContext.getUrlAliasMgr().setAliasAbsPath(aliasName, aliasPath)

            # Progress
            xmlElmtProgress = xmlElmtContext.find('progress')
            progressIndex = xmlElmtProgress.get('index')
            progressCount = xmlElmtProgress.get('count')
            try:
                self.mSBSContext.setProgress(int(progressIndex), int(progressCount))
            except:
                self.mSBSContext.setProgress(0, 0)

        # Build function arguments
        self.mFctArgs = {}
        xmlElmtArgs = self.mXMLRoot.find('fct_args')
        if xmlElmtArgs is not None:
            # Args
            for xmlElmtChild in xmlElmtArgs:
                if not xmlElmtChild.tag == 'arg':
                    continue
                argName = xmlElmtChild.get('name')
                if argName is None:
                    continue
                argValue = xmlElmtChild.get('value')
                if argValue is None:
                    continue
                self.mFctArgs[argName] = argValue

        return True

#==============================================================================
class CommandLineArgsProcessor:
    def __init__(self, aCallerModuleName = __name__):
        self.mCallerPyModule = sys.modules[aCallerModuleName]

    def call(self):
        context = CommandLineArgs()
        context.parseArgs(sys.argv[1:])
        exitCode = self.__execute(context)
        sys.exit(exitCode)

    #==============================================================================
    def __execute(self, aContext):
        fctName = aContext.getArgument("fct")
        argsFileAbsPath = aContext.getArgument("args")
        retValFile = aContext.getArgument("retval")

        if fctName is None or len(fctName) == 0:
            print('Function name empty')
            return 0

        if self.mCallerPyModule is None:
            return 0

        if not hasattr(self.mCallerPyModule, fctName):
            print('Function "'+str(fctName)+'" not found')
            return 0
        fct = getattr(self.mCallerPyModule, fctName)

        argsFile = ArgsFile(argsFileAbsPath)
        if not argsFile.parse():
            print('Fail to parse argument file :"'+argsFileAbsPath+'"')
        fctArgsDict = argsFile.getFunctionArgs()
        fctArgsDict['aContext'] = argsFile.getSBSContext()

        # call fct
        print('Calling fct "'+fctName+'" with args '+str(fctArgsDict))
        returnValue = fct(**fctArgsDict)

        # write return value to retValFile
        if returnValue is not None:
            xmlRoot = ET.Element('return_value')
            if xmlRoot is None:
                print('Fail to create return value xml doc')

            if isinstance(returnValue, ReturnValue):
                for key in returnValue.getKeys():
                    attributes = {}
                    attributes['name'] = key
                    attributes['value'] = returnValue.getValue(key)
                    ET.SubElement(xmlRoot, 'data', attributes)
            else:
                xmlRoot.text = str(returnValue)

            fileContent = ET.tostring(xmlRoot, encoding="utf-8")

            f = open(retValFile, 'wt')
            if f is not None:
                f.write(fileContent)
                f.close()

        return 1
